function values = value_default_leg_tranches(cut_offs, T, coupons, upfront, discounts, discounts_inbetween, loss_dist, ...
                                             num_available, LGD_mean, LGD_method, alpha_n_rel, G_n_storage)
% --------------------------------------------------------------------------------------------------
% Calculate the market value of the payments made by the seller of protection for each tranche.
% --------------------------------------------------------------------------------------------------
% cut_offs              ... cutoff points in tranche structure
% T                     ... time to maturity of index
% coupons               ... coupon rates for individual tranches
% upfront               ... upfront payments for tranches
% discounts             ... discount curves (matching coupon payments, end of period)
% discounts_inbetween   ... discount curves (matching coupon payments, middle of period)
% loss_dist             ... 3-dim array, entry (i,j,k) gives the probability of (k-1) defaults on
%                           the i-th date, for the j-th time horizon
% num_available         ... number of firms in portfolio at each point in time
% LGD_mean              ... mean LGD
% LGD_method            ... method for (joint) distribution of LGDs
% alpha_n_rel           ... (relative) support/2 for LGD_method two
% G_n_storage           ... port. loss. dist. cond. on number of defaults
% --------------------------------------------------------------------------------------------------
% sample call: value_default_leg_tranches(cdx_test.cut_offs, [5; 4.95], ...)
% --------------------------------------------------------------------------------------------------

% Compute market value of default payments for each tranche
num_dates = length(T);
num_tranches = length(cut_offs);
values = zeros(num_dates, num_tranches);
cut_offs = [0 cut_offs];
for i=1:num_dates
    num_firms = num_available(i);
    for t=1:ceil(T(i)*4)
        % Extract portfolio loss distribution at time t-1 and t
        loss_dist_t = reshape(loss_dist(i, t+1, 1:(num_firms+1)), 1, num_firms+1);
        loss_dist_t_old = reshape(loss_dist(i, t, 1:(num_firms+1)), 1, num_firms+1);

        % Compute difference in porfolio loss distribution between time t-1 and t
        if (LGD_method == 0)        % deterministic LGD
            delta_loss_density = loss_dist_t - loss_dist_t_old;
        elseif (LGD_method == 1)    % P(LGD_i=0.45)=2/3, P(LGD_i=0.9)=1/3 and iid LGDs
            % Calculate loss density
            delta_loss_density = zeros(1, num_firms*2+1);
            delta_num_loss = loss_dist_t - loss_dist_t_old;
            for n=1:length(delta_num_loss)
               G_n = G_n_storage{n};
               delta_loss_density(n:(2*n-1)) = delta_loss_density(n:(2*n-1)) + G_n * delta_num_loss(n);
            end
        elseif (LGD_method == 2)    % G_n has uniform distribution
            delta_loss_density = loss_dist_t - loss_dist_t_old;
            % Cut down alpha_n to appropriate length
            alpha_n = alpha_n_rel(1:(num_firms+1));
            alpha_n = alpha_n .* (0:num_firms)/num_firms;
        elseif (LGD_method == 3)    % LGD_n follows a Markov chain
            % Calculate loss density
            delta_loss_density = zeros(1, num_firms*3+1);
            delta_num_loss = loss_dist_t - loss_dist_t_old;
            for n=1:length(delta_num_loss)
               G_n = G_n_storage{n};
               delta_loss_density(n:(3*n-2)) = delta_loss_density(n:(3*n-2)) + G_n * delta_num_loss(n);
            end            
        end
            
        % Attribute losses to tranches
        for j=1:num_tranches
            % Compute expected tranche size loss between time t-1 and time t
            if (LGD_method == 0)        % deterministic LGD
                loss_amounts = (0:num_firms) * LGD_mean / num_firms;
                notional = min( cut_offs(j+1)-cut_offs(j), max(0, cut_offs(j+1)-loss_amounts) );
                exp_tranche_size_loss = (-1) * delta_loss_density * notional';
            elseif (LGD_method == 1)    % P(LGD_i=0.45)=2/3, P(LGD_i=0.9)=1/3 and iid LGDs
                loss_amounts = (0:(num_firms*2)) * 0.45 / num_firms;
                notional = min( cut_offs(j+1)-cut_offs(j), max(0, cut_offs(j+1)-loss_amounts) );
                exp_tranche_size_loss = (-1) * delta_loss_density * notional';
            elseif (LGD_method == 2)
                upper = (0:num_firms)*LGD_mean/num_firms + alpha_n;
                lower = (0:num_firms)*LGD_mean/num_firms - alpha_n;
                integral_bound_upper = max(min(upper, cut_offs(j+1)), cut_offs(j));
                integral_bound_lower = min(max(lower, cut_offs(j)), cut_offs(j+1));
                integral_part1 = 1/2 * ((integral_bound_upper-cut_offs(j)).^2 - (integral_bound_lower-cut_offs(j)).^2);
                integral_part2 = (max(upper - cut_offs(j+1), 0) - max(lower - cut_offs(j+1), 0)) * (cut_offs(j+1)-cut_offs(j));
                integral_total = 1./(2*alpha_n(2:end)) .* (integral_part1(2:end) + integral_part2(2:end));
                integral_total = [0 integral_total];
                exp_tranche_size_loss = sum(delta_loss_density .* integral_total);
            elseif (LGD_method == 3)    % LGD_n follows a Markov chain
                loss_amounts = (0:(num_firms*3)) * 0.3 / num_firms;
                notional = min( cut_offs(j+1)-cut_offs(j), max(0, cut_offs(j+1)-loss_amounts) );
                exp_tranche_size_loss = (-1) * delta_loss_density * notional';
            end
            
            % Add discounted default payment to market value of tranche
            values(i,j) = values(i,j) + exp_tranche_size_loss * discounts_inbetween(i,t);
            
            % Substract arrued premium at time of default
            values(i,j) = values(i,j) - exp_tranche_size_loss * discounts_inbetween(i,t) * coupons(i,j)/1e4/2 * (365/360);
        end
    end
end
%disp(values);

